import { Router } from 'express';
import { sketchesController } from './controller';
import { authenticateUser, requireVendor } from '@/middlewares/auth';
import { checkFeature } from '@/middlewares/featureGate';
import { vendorService } from '@/modules/vendors/service';
import multer from 'multer';

const upload = multer();
const router = Router();

// Attach vendor to req for downstream use
async function attachVendor(req: any, res: any, next: any) {
  if (req.user) {
    const { data: vendor } = await vendorService.getVendorByUserId(req.user.id);
    req.vendor = vendor || null;
  }
  next();
}

// Device guard middleware for SVG enhancement
function deviceGuard(req: any, res: any, next: any) {
  // Assume device info is in req.device (set by frontend or earlier middleware)
  if (req.body.enhancement_type === 'svg' && req.device !== 'desktop' && req.device !== 'tablet') {
    return res.status(403).json({ error: 'SVG enhancement only allowed on desktop/tablet.' });
  }
  next();
}

// Upload sketch (multipart)
router.post('/', authenticateUser, requireVendor, attachVendor, checkFeature('sketch_upload'), upload.single('file'), sketchesController.upload);
// Request AI enhancement
router.post('/:id/enhance', authenticateUser, requireVendor, attachVendor, checkFeature('ai_sketch_enhance'), deviceGuard, sketchesController.enhance);
// List sketches
router.get('/', authenticateUser, requireVendor, attachVendor, sketchesController.list);
// Get single sketch
router.get('/:id', authenticateUser, requireVendor, attachVendor, sketchesController.get);
// Delete sketch
router.delete('/:id', authenticateUser, requireVendor, attachVendor, sketchesController.delete);

export default router; 